-- MinimapSla by SLA
-- Created: 2010-03-18
-- Updated: 2011-06-26
-- http://ui9.ru/
--------------------------------------------------------------------------------
-- GLOBALS
--------------------------------------------------------------------------------
Global( "Minimap", {} )

Global( "onBase", {} )
Global( "onEvent", {} )
Global( "onIncognita", {} )

Global( "onReaction", {} )

Global( "wtTooltip", nil )
Global( "wtMainPanel", nil )
Global( "wtButtons", nil )
Global( "wtZoomIn", nil )
Global( "btnZoomIn", nil )
Global( "wtZoomOut", nil )
Global( "btnZoomOut", nil )
Global( "wtPointers", nil )
Global( "wtPlayerArrow", nil )
Global( "wtMarkers", nil )
Global( "wtHighlighter", nil )
Global( "wtMarkerDESC", nil )
Global( "wtMouseCatcherDESC", nil )
Global( "wtCircle", nil )
Global( "wtMap", nil )
Global( "wtInstance", nil )
Global( "wtTitle", nil )
Global( "labTitle", nil )

Global( "enumMode", { Disabled = 0, Exploring = 1, Instance = 2, Standard = 3 } )
Global( "Mode", enumMode.Disabled )

Global( "MapID", nil )
Global( "MapGeodata", nil )
Global( "MapMPP", nil )

Global( "MapExplore", nil )
Global( "iSafeEnter", 2 )
Global( "iSafeQuit", 30 )

Global( "MapRadius", nil )
Global( "MapPlacement", nil )

Global( "ZoomSize", {} )
Global( "Zoom", 1 )
Global( "fFadeoutButtons", false )

Global( "HoveredMarker", nil )

Global( "Markers", {
	SOS = {},
	Spouse = {},
	Party = {},
	Track = {},
	Transport = {},
	Quest = {},
	Service = {},
	MapMark = {},
	Extra = {},
} )

Global( "MarkersPriority", { -- Also, the list of valid icons.
	-- Active:
	SOS = 86,
	Spouse = 85,
	Party = 84,
	Raid = 83,
	Track = 82,
	Transport = 81,
	-- Quest (NPCs):
	QuestDoneSecret = 68,
	QuestDone = 67,
	QuestDoneRepeat = 66,
	QuestDoneTutorial = 66,
	QuestHasSecret = 65,
	QuestHas = 64,
	QuestHasRepeat = 63,
	QuestHasTutorial = 63,
	QuestProgress = 62,
	QuestUnavailable = 61,
	Quest = 60,
	-- Service:
	Mug = 52,
	Mail = 51,
	Mail1 = 51,
	Mail2 = 51,
	Teleport = 50,
	Guild = 49,
	Remort = 49,
	Exchange = 49,
	ChangeRoom = 49,
	Trainer = 49,
	Auction = 48,
	Safe = 48,
	HonorEmpire = 47,
	HonorLeague = 47,
	Reputation = 46,
	Myrrh = 45,
	Crafting = 44,
	Consumable = 43,
	Weapon = 42,
	Trader = 41,
	Service = 40,
	-- Quest (Locations):
	QuestPlaceItem = 31,
	QuestPlaceKill = 31,
	QuestPlaceSpecial = 31,
	QuestPlaceReturn = 30,
	-- MapMark:
	MapMark = 10,
	-- Extra:
	Crystal01 = 6,
	Crystal02 = 6,
	Crystal03 = 6,
	Boss = 4,
	Treasure = 3,
	Milestone = 2,
	Placeholder = 1,
	Extra = 0,
} )

Global( "MarkersScale", {
	Default = 0.5,
	Track = 0.5,
	MapMark = 0.67,
} )

Global( "ActiveAreaScale", {
	Default = 0.6,
	Party = 1.0,
	Raid = 1.0,
	MapMark = 0.4,
} )

Global( "Style", {
	Friend = "Friendly",
	Neutral = "Neutral",
	Enemy = "Aggressive",
	Party = "LogColorBlue",
	Raid = "LogColorOrange",
	Spouse = "LogColorPink",
	Quest = "tip_golden",
	Blank = "tip_white",
	Minor = "tip_grey",
} )

Global( "ClassColors", {
	["WARRIOR"]		= { r = 0.65; g = 0.54; b = 0.34; a = 1 },
	["PALADIN"]		= { r = 0.00; g = 0.88; b = 0.78; a = 1 },
	["STALKER"]		= { r = 0.00; g = 0.78; b = 0.00; a = 1 },
	["PRIEST"]		= { r = 1.00; g = 1.00; b = 0.31; a = 1 },
	["DRUID"]		= { r = 1.00; g = 0.50; b = 0.00; a = 1 },
	["PSIONIC"]		= { r = 1.00; g = 0.50; b = 1.00; a = 1 },
	["MAGE"]		= { r = 0.18; g = 0.57; b = 1.00; a = 1 },
	["NECROMANCER"]	= { r = 0.95; g = 0.17; b = 0.28; a = 1 },
	["UNKNOWN"]		= { r = 0.50; g = 0.50; b = 0.50; a = 1 },
} )

--------------------------------------------------------------------------------
-- REACTION HANDLERS
--------------------------------------------------------------------------------
onReaction[ "zoom" ] = function( params )
	if params.sender == "btnZoomIn" then
		Zoom = math.min( Zoom + 1, table.getn( ZoomSize ) )
	else -- "btnZoomOut"
		Zoom = math.max( Zoom - 1, 1 )
	end
	btnZoomIn:Enable( Zoom < table.getn( ZoomSize ) )
	btnZoomOut:Enable( Zoom > 1 )
	MapPlacement.sizeX = ZoomSize[Zoom].X
	MapPlacement.sizeY = ZoomSize[Zoom].Y
	wtMap:SetPlacementPlain( MapPlacement )
	wtMarkers:SetPlacementPlain( MapPlacement )
	onBase[ "EVENT_AVATAR_CLIENT_ZONE_CHANGED" ]()
end
--------------------------------------------------------------------------------
onReaction[ "show_buttons" ] = function( params )
	wtButtons:FinishFadeEffect()
	if params.active then
		if wtButtons:IsVisible() then
			fFadeoutButtons = false
			wtButtons:SetFade( 1.0 )
		else
			wtButtons:Show( true )
			wtButtons:PlayFadeEffect( 0.0, 1.0, 300, EA_MONOTONOUS_INCREASE )
		end
	else
		if wtButtons:IsVisible() then
			fFadeoutButtons = true
			wtButtons:PlayFadeEffect( 1.0, 0.0, 300, EA_MONOTONOUS_INCREASE )
		end
	end
end
--------------------------------------------------------------------------------
-- Marker Tooltip
onReaction[ "tooltip" ] = function( params )
	if params.active then
		-- Finding marker
		local ID = tonumber( string.sub( params.sender, string.len( "Mark" ) + 1 ) )
		if not ID then return end
		local Marker
		for L,M in Markers do
			if M[ ID ] then
				Marker = M[ ID ]
				break
			end
		end
		if not Marker then return end
		-- Tooltip strings setup
		local wtText = wtTooltip:GetChildChecked( "TooltipText", false )
		wtText:ClearValues()
		wtText:SetClassVal( "style", Marker.Style )
		for i in Marker.Name do
			if i == 9 then break end
			wtText:SetVal( "name" .. i, Marker.Name[ i ] )
			wtText:SetVal( "desc" .. i, Marker.Desc[ i ] )
		end
		-- Tooltip placement
		local ir = Marker.Widget:GetRealRect()
		local tr = wtTooltip:GetRealRect()
		local p = wtTooltip:GetPlacementPlain()
		p.posX = math.floor( ir.x1 - ( tr.x2 - tr.x1 ) )
		p.posY = math.floor( ir.y1 )
		wtTooltip:SetPlacementPlain( p )
		wtTooltip:Show( true )
		-- Highlighter creation
		HoveredMarker = Marker.ID
		wtHighlighter:SetBackgroundTexture( Marker.Texture )
		wtHighlighter:SetPlacementPlain( Marker.Widget:GetPlacementPlain() )
		wtHighlighter:SetBackgroundColor( Marker.Widget:GetBackgroundColor() )
		wtHighlighter:Rotate( Marker.Widget:GetRotation() )
		wtHighlighter:Show( true )
	else
		wtTooltip:Show( false )
		wtHighlighter:Show( false )
		HoveredMarker = nil
	end
end
--------------------------------------------------------------------------------
-- EVENT HANDLERS
--------------------------------------------------------------------------------
-- Purging Opposite Faction resources
onBase[ "EVENT_AVATAR_CREATED" ] = function( params )
	if AoMapResources.PurgeUselessResources then
		AoMapResources:PurgeUselessResources()
	end
	onBase[ "SCRIPT_ADDON_INFO_REQUEST" ]( { target = common.GetAddonName() } )
	-- OMG, this event doesn't fire when relogging with the same character:
	onBase[ "EVENT_AVATAR_CLIENT_ZONE_CHANGED" ]()
end
--------------------------------------------------------------------------------
-- Zone changed (full reinitialization)
onBase[ "EVENT_AVATAR_CLIENT_ZONE_CHANGED" ] = function( params )
	EventHandlersRegistration( false, onEvent )
	EventHandlersRegistration( false, onIncognita )
	
	local map = cartographer.GetCurrentMap()
	local zone = cartographer.GetCurrentZone()
	local zonesMapInfo = cartographer.GetZonesMapInfo( zone.zonesMapId )
	local ShipID = unit.GetTransport( avatar.GetId() )
	local fShipHasUI = ShipID and transport.CanDrawInterface( ShipID ) or false
	
	--if fShipHasUI or not map or not map.isTerrain or not zone or not zone.zonesMapId or not zonesMapInfo then
	if fShipHasUI or not map or not zone or not zone.zonesMapId or not zonesMapInfo then
		Mode = enumMode.Disabled
	elseif not zonesMapInfo.isUnknown and zonesMapInfo.isShowPositions and zonesMapInfo.texture and
	not AoMapResources:InstanceMapFound( true, zonesMapInfo.sysName, zone.sysSubZoneName ) then
		Mode = enumMode.Standard
	elseif AoMapResources:InstanceMapFound( false, zone.sysZoneName, zone.sysSubZoneName ) then
		Mode = enumMode.Instance
	else
		Mode = enumMode.Exploring
	end
	
	if Mode == enumMode.Standard then
		MapGeodata = cartographer.GetObjectGeodata( avatar.GetId(), zone.zonesMapId )
		if not MapGeodata then Mode = enumMode.Disabled end
	end
	
	-- Those 2 names sometimes differ. I prefer first one, because it is universal.
	local strSysZoneName = ( Mode == enumMode.Standard) and zonesMapInfo.sysName or zone.sysZoneName
	
	-- If there is data collected in previous instance, lets save it
	if MapExplore and ( Mode ~= enumMode.Exploring or MapExplore.Zone ~= strSysZoneName ) then
		local CurPos = object.GetPos( avatar.GetId() )
		local Bounds = MapExplore.B[ iSafeQuit ]
		-- If this double protection fails, either restrict bounds collection
		-- to avatar only, or completely disable saving/loading.
		if Bounds and not Bounds.Dirty and
		Bounds.MaxX + 100 < CurPos.posX and Bounds.MaxY + 100 < CurPos.posY then
			local cfg = userMods.GetGlobalConfigSection( common.GetAddonName() ) or {}
			if not cfg.Bounds then cfg.Bounds = {} end
			cfg.Bounds[ MapExplore.Zone ] = string.format( "%.3f %.3f %.3f %.3f",
				Bounds.MinX, Bounds.MaxX, Bounds.MinY, Bounds.MaxY )
			userMods.SetGlobalConfigSection( common.GetAddonName(), cfg )
		end
		MapExplore = nil
	end
	
	MapID = zone.zonesMapId
	EventHandlersRegistration( Mode ~= enumMode.Disabled, onEvent )
	EventHandlersRegistration( Mode == enumMode.Exploring, onIncognita )
	wtMainPanel:Show( Mode ~= enumMode.Disabled )
	if Mode == enumMode.Disabled then return end
	
	-- Painting Title
	if not common.IsEmptyWString( zone.subZoneName ) then
		labTitle:SetVal( "value", zone.subZoneName )
	else
		labTitle:SetVal( "value", zone.zoneName )
	end
	
	-- Painting Map
	if Mode == enumMode.Standard then -- Normal map
		wtMap:SetBackgroundTexture( zonesMapInfo.texture )
		--MapGeodata = cartographer.GetObjectGeodata( avatar.GetId(), MapID )
	else
		local TextureData
		local TextureRatio
		if Mode == enumMode.Instance then -- Known Instance :)
			local Instance = AoMapResources:GetInstanceMap( strSysZoneName, zone.sysSubZoneName )
			TextureData = common.GetAddonRelatedTexture( Instance.Texture.name )
			TextureRatio = Instance.Texture.ratio
			MapGeodata = Instance.Geodata
		else -- enumMode.Exploring -- Unknown Instance
			TextureData = common.GetAddonRelatedTexture( "_Unknown" )
			TextureRatio = 1
			if not MapExplore then
				MapExplore = {}
				MapExplore.Zone = strSysZoneName
				MapExplore.Altered = true
				MapExplore.B = {}
			end
			if not MapExplore.B[ iSafeEnter ] then -- Forced Resets on first iSafeEnter seconds in instance.
				local cfg = userMods.GetGlobalConfigSection( common.GetAddonName() )
				if cfg and cfg.Bounds and cfg.Bounds[ strSysZoneName ] then
					-- Instance Bounds, loading from config.
					local un = {}
					for v in string.gfind( cfg.Bounds[ strSysZoneName ], "[%-%d%.]+" ) do
						table.insert( un, tonumber( v ) )
					end
					MapExplore.B[0] = {
						MinX = un[ 1 ], MaxX = un[ 2 ],
						MinY = un[ 3 ], MaxY = un[ 4 ],
						Dirty = true
					}
				else
					-- Instance Bounds, creating new.
					local Start = object.GetPos( avatar.GetId() )
					MapExplore.B[0] = {
						MinX = Start.posX - 1, MaxX = Start.posX + 1,
						MinY = Start.posY - 1, MaxY = Start.posY + 1,
						Dirty = true
					}
				end
				local DetectedObjects = avatar.GetDetectedObjects()
				for _,ID in DetectedObjects do
					if not object.IsUnit( ID ) or object.IsUnit( ID ) and not unit.IsPlayer( ID ) then
						Minimap.CheckBounds( object.GetPos( ID ), 300 )
					end
				end
				MapGeodata = {}
				Minimap.RecalculateGeodata()
			end
		end
		wtMap:SetBackgroundTexture( common.GetAddonRelatedTexture( "_Blank" ) )
		wtInstance:SetBackgroundTexture( TextureData )
		local tsize = widgetsSystem.GetTextureSize( TextureData )
		local p = wtInstance:GetPlacementPlain()
		p.sizeX = math.floor( tsize.sizeX / TextureRatio / 1024 * MapPlacement.sizeX )
		p.sizeY = math.floor( tsize.sizeY / TextureRatio / 886 * MapPlacement.sizeY )
		p.posX = math.floor( ( MapPlacement.sizeX - p.sizeX ) / 2 )
		p.posY = math.floor( ( MapPlacement.sizeY - p.sizeY ) / 2 )
		wtInstance:SetPlacementPlain( p )
	end
	wtInstance:Show( Mode ~= enumMode.Standard )
	MapMPP = { X = MapGeodata.width / ZoomSize[Zoom].X, Y = MapGeodata.height / ZoomSize[Zoom].Y }
	
	-- Got all required data (MapID, MapGeodata, MapMPP). Now placing markers.
	
	Minimap.DestroyMarker() -- Clears all markers.
	
	-- Known Objects markers :)
	local ZoneObjects = AoMapResources:GetKnownObjects( strSysZoneName, MapGeodata )
	if ZoneObjects then
		local FakeID = -1000
		for _,obj in ZoneObjects do
			local Color = Style.Friend
			if obj.Type == "Boss" then Color = Style.Enemy end
			if obj.Type == "Milestone" or obj.Type == "Treasure" then Color = Style.Minor end
			FakeID = FakeID - 1
			Minimap.CreateMarker( "Extra", FakeID, AoMapResources:L( obj.Type ), AoMapResources:L( obj.Desc ), Color, obj.Type, MapGeodata, obj.Pos )
		end
	end
	-- Map markers
	if cartographer.GetMapMarkers then -- AO 2.0.07+ only
		local mapMarkers = cartographer.GetMapMarkers( MapID )
		local FakeID = -2000
		for _,ID in mapMarkers do
			local mi = cartographer.GetMarkerInfo( ID )
			local mo = cartographer.GetMapMarkerObjects( MapID, ID )
			for _,obj in mo do
				FakeID = FakeID - 1
				mi.description = StripHtmlTags( mi.description )
				Minimap.CreateMarker( "MapMark", FakeID, mi.name, mi.description, Style.Blank, mi.image, obj.geodata, obj.pos )
			end
		end
	end
	-- Spouse marker
	if mount.SetSkin then -- AO 2.0.01+ only
		if family.IsExist( avatar.GetId() ) and family.GetSpouseId( avatar.GetId() ) then
			local SpouseID = family.GetSpouseId( avatar.GetId() )
			local Geodata = Minimap.ForceGetObjectGeodata( SpouseID )
			Minimap.CreateMarker( "Spouse", SpouseID, family.GetSpouseName( avatar.GetId() ), nil, Style.Spouse, nil, Geodata, nil )
		end
	end
	-- Detector markers
	local detectedObjects = avatar.GetDetectedObjects()
	for _,ID in detectedObjects do
		onEvent[ "EVENT_OBJECT_DETECT_STARTED" ]( { objectId = ID, GUARANTEED = true } )
	end
	-- Track markers
	local trackedObjects = objects.GetTracks()
	for _,ID in trackedObjects do
		onEvent[ "EVENT_TRACK_ADDED" ]( { id = ID } )
	end
	-- Transport markers
	local transportList = avatar.GetTransportList()
	for _,ID in transportList do
		onEvent[ "EVENT_TRANSPORT_SPAWNED" ]( { id = ID } )
	end
	-- All other markers, scrolling map, rotating arrow.
	onEvent[ "EVENT_GROUP_CHANGED" ]()
	onEvent[ "EVENT_QUEST_BOOK_CHANGED" ]()
	onEvent[ "EVENT_AVATAR_DIR_CHANGED" ]()
	onEvent[ "EVENT_AVATAR_POS_CHANGED" ]()
end
onBase[ "EVENT_AVATAR_TRANSPORT_CHANGED" ] = onBase[ "EVENT_AVATAR_CLIENT_ZONE_CHANGED" ]
--------------------------------------------------------------------------------
-- Quests changed
onEvent[ "EVENT_QUEST_BOOK_CHANGED" ] = function( params )
	Minimap.DestroyMarker( "Quest", nil )
	local Collection = {}
	local Quests = avatar.GetQuestBook()
	for _,Q in Quests do
		local qg = cartographer.GetQuestGeodata( Q, MapID )
		local qi = avatar.GetQuestInfo( Q )
		local qp = avatar.GetQuestProgress( Q )
		
		-- A short introduction to AO QuestBook data (just a reminder for myself):
		-- Each quest SHOULD have qi.returnLocation (afaik, all quests have).
		-- If it have qi.returnLocation, it CAN have qg.returnGeodata on current map.
		-- Many quests CAN have qi.goalLocation.
		-- If it have qi.goalLocation, it CAN have qg.goalGeodata on current map.
		-- Some quests which have qi.goalLocation, CAN also have non-empty qi.additionalLocations[].
		-- If it have non-empty qi.additionalLocations[], it CAN have non-empty qg.additionalGeodata[],
		-- with only SOME indexes from qi.additionalLocations[] filled in, when geodata present on current map.
		-- In instances, there is NO qg.*Geodata at all, but there is qi.*Location.zonesMapId, which is enaugh for instances.
		-- qp.objectives[] are *NOT* RELATED to locations at all, but qp.objectives[0] always exists, and it defines the icon type.
		
		if qp.state == QUEST_READY_TO_RETURN then
			local Icon = "QuestPlaceReturn"
			local Geodata = qg.returnGeodata or
				qi.returnLocation and qi.returnLocation.zonesMapId == MapID and MapGeodata
			if Geodata then
				local X = qi.returnLocation.position.posX
				local Y = qi.returnLocation.position.posY
				if not Collection[ X ] then Collection[ X ] = {} end
				if not Collection[ X ][ Y ] then Collection[ X ][ Y ] = {} end
				table.insert( Collection[ X ][ Y ], { N=qi.name, D=qi.goal, I=Icon, G=Geodata } )
			end
		elseif qp.state == QUEST_IN_PROGRESS then
			local Icon = "QuestPlaceSpecial" -- QUEST_COUNT_SPECIAL, etc.
			if qp.objectives[ 0 ].type == QUEST_COUNT_KILL then Icon = "QuestPlaceKill" end
			if qp.objectives[ 0 ].type == QUEST_COUNT_ITEM then Icon = "QuestPlaceItem" end
			local Geodata = qg.goalGeodata or
				qi.goalLocation and qi.goalLocation.zonesMapId == MapID and MapGeodata
			if Geodata then
				local X = qi.goalLocation.position.posX
				local Y = qi.goalLocation.position.posY
				if not Collection[ X ] then Collection[ X ] = {} end
				if not Collection[ X ][ Y ] then Collection[ X ][ Y ] = {} end
				table.insert( Collection[ X ][ Y ], { N=qi.name, D=qi.goal, I=Icon, G=Geodata } )
			end
--			for i in qi.additionalLocations do
--				local Geodata = qg.additionalGeodata and qg.additionalGeodata[ i ] or
--					qi.additionalLocations[ i ].zonesMapId == MapID and MapGeodata
--				if Geodata then
--					local X = qi.additionalLocations[ i ].position.posX
--					local Y = qi.additionalLocations[ i ].position.posY
--					if not Collection[ X ] then Collection[ X ] = {} end
--					if not Collection[ X ][ Y ] then Collection[ X ][ Y ] = {} end
--					table.insert( Collection[ X ][ Y ], { N=qi.name, D=qi.goal, I=Icon, G=Geodata } )
--				end
--			end
		end
		-- Also, in Unknown instance, checking all hidden quest locations.
		if Mode == enumMode.Exploring then
			if qi.returnLocation and qi.returnLocation.zonesMapId == MapID then
				Minimap.CheckBounds( qi.returnLocation.position, 1000 )
			end
			if qi.goalLocation and qi.goalLocation.zonesMapId == MapID then
				Minimap.CheckBounds( qi.goalLocation.position, 1000 )
			end
--			for i in qi.additionalLocations do
--				if qi.additionalLocations[ i ].zonesMapId == MapID then
--					Minimap.CheckBounds( qi.additionalLocations[ i ].position, 1000 )
--				end
--			end
		end
	end
	-- All quest data collected, now placing markers.
	local FakeID = 0
	for X in Collection do
		for Y in Collection[ X ] do
			local Icon = Collection[ X ][ Y ][ 1 ].I
			local Geodata = Collection[ X ][ Y ][ 1 ].G
			local tabNames = {}
			local tabDescs = {}
			for _,v in Collection[ X ][ Y ] do -- Because one marker can hold several quests.
				v.D = StripHtmlTags( v.D )
				table.insert( tabNames, v.N or false )
				table.insert( tabDescs, v.D or false )
			end
			FakeID = FakeID - 1
			Minimap.CreateMarker( "Quest", FakeID, tabNames, tabDescs, Style.Quest, Icon, Geodata, { posX = X, posY = Y } )
		end
	end
end
--------------------------------------------------------------------------------
-- Object detected
onEvent[ "EVENT_OBJECT_DETECT_STARTED" ] = function( params )
	-- This event fires for Party members too, but I need only NPCs/Objects here.
	if object.IsUnit( params.objectId ) and unit.IsPlayer( params.objectId ) then return end
	
	-- Sometimes, the server tries to cheat me.
	if not params.GUARANTEED then
		local fFound
		local DetectedObjects = avatar.GetDetectedObjects()
		for _,ID in DetectedObjects do
			if ID == params.objectId then
				fFound = true
				break
			end
		end
		if not fFound then return end
	end
	
	local Pos = object.GetPos( params.objectId )
	local ii = object.GetInteractorInfo( params.objectId )
	if not Pos or not ii then return end
	local Name = object.GetName( params.objectId )
	local Desc = object.IsUnit( params.objectId ) and unit.GetTitle( params.objectId ) or nil
	local Icon = "Placeholder"
	local Color = Style.Friend
	Minimap.CheckBounds( Pos, 300 )
	
	if object.IsUnit( params.objectId ) then
		if unit.IsFriend( params.objectId ) then
			Color = Style.Friend
		elseif unit.IsEnemy( params.objectId ) then
			Color = Style.Enemy
		else
			Color = Style.Neutral
		end
	end
	
	-- Questgiver?
	if     ii.isQuestGiver then
		local qm = object.GetQuestMark( params.objectId ) or QUEST_MARK_SOON_TO_GIVE
		if     qm == QUEST_MARK_READY_TO_ACCEPT then Icon = "QuestDone"
		elseif qm == QUEST_MARK_SOON_TO_ACCEPT then Icon = "QuestProgress"
		elseif qm == QUEST_MARK_READY_TO_GIVE then Icon = "QuestHas"
		elseif qm == QUEST_MARK_SOON_TO_GIVE then Icon = "QuestUnavailable"
		end
		if qm == QUEST_MARK_READY_TO_ACCEPT or qm == QUEST_MARK_READY_TO_GIVE then
			if ii.isSecretRelated or ii.isSecretFinisher then
				Icon = Icon .. "Secret"
			elseif object.HasOnlyRepeatableQuests( params.objectId ) then
				Icon = Icon .. "Repeat"
			end
		end
	-- Service?
	elseif ii.isTrainer then Icon = "Trainer"
	elseif ii.isMailBox then Icon = "Mail"
	elseif ii.isAuction then Icon = "Auction"
	elseif ii.isTeleportMaster then Icon = "Teleport"
	elseif ii.isRemortMaster then Icon = "Remort"
	elseif ii.isDepositeBoxAccessor then Icon = "Safe"
	elseif ii.isCurrencyExchanger then Icon = "Exchange"
	elseif ii.isChangeRoomMaster then Icon = "ChangeRoom"
	elseif ii.canTakeRestedExp then Icon = "Mug"
	elseif ii.extended[ INTERACTION_BINDING_STONE ] then Icon = "Teleport"
	elseif ii.extended[ INTERACTION_CLASS_RELATED ] then Icon = "Trainer"
	-- Vendor?
	elseif ii.isReputationVendor then Icon = "Reputation"
	elseif ii.isVendor then
		if     ii.vendorType == VENDOR_WEAPON_ARMOR then Icon = "Weapon"
		elseif ii.vendorType == VENDOR_GUILD then Icon = "Guild"
		elseif ii.vendorType == VENDOR_CRAFTING_COMPONENTS then Icon = "Crafting"
		elseif ii.vendorType == VENDOR_USABLE_ITEMS then Icon = "Consumable"
		elseif ii.vendorType == VENDOR_MYRRH then Icon = "Myrrh"
		elseif ii.isHonorVendor then Icon = "Honor" .. unit.GetFaction( avatar.GetId() ).sysName
		else Icon = "Trader" -- VENDOR_GENERAL
		end
	end
	
	Minimap.CreateMarker( "Service", params.objectId, Name, Desc, Color, Icon, MapGeodata, Pos )
end
--------------------------------------------------------------------------------
onEvent[ "EVENT_OBJECT_DETECTOR_LIST_CHANGED" ] = function( params )
	onEvent[ "EVENT_OBJECT_DETECT_FINISHED" ]( params )
	onEvent[ "EVENT_OBJECT_DETECT_STARTED" ]( params )
end
onEvent[ "EVENT_QUEST_MARK_UPDATED" ] = onEvent[ "EVENT_OBJECT_DETECTOR_LIST_CHANGED" ]
--------------------------------------------------------------------------------
-- Object lost
onEvent[ "EVENT_OBJECT_DETECT_FINISHED" ] = function( params )
	Minimap.DestroyMarker( "Service", params.objectId )
end
--------------------------------------------------------------------------------
-- Party / Raid changed
onEvent[ "EVENT_GROUP_CHANGED" ] = function( params )
	Minimap.DestroyMarker( "Party", nil )
	if not group.GetMembers() and not raid.IsExist() then return end
	-- Collecting party/raid members
	local Members
	if not raid.IsExist() then
		Members = group.GetMembers() or {}
	else
		Members = {}
		local RaidGroups = raid.GetMembers()
		for _, Group in RaidGroups do
			for _, member in Group do
				table.insert( Members, member )
			end
		end
	end
	-- Creating member markers
	for _, member in Members do
		if member.state == GROUP_MEMBER_STATE_NEAR and member.id and member.id ~= avatar.GetId() then
			local Icon, Color
			if raid.IsExist() and not raid.IsPlayerInAvatarsRaidGroup( member.name ) then
				Icon, Color = "Raid", Style.Raid
			else
				Icon, Color = "Party", Style.Party
			end
			-- Afaik, all NEAR party/raid members are inside the instance :)
			local Geodata = ( Mode == enumMode.Standard ) and cartographer.GetObjectGeodata( member.id, MapID ) or MapGeodata
			Minimap.CreateMarker( "Party", member.id, member.name, nil, Color, Icon, Geodata, nil )
		end
	end
end
onEvent[ "EVENT_GROUP_APPEARED" ] = onEvent[ "EVENT_GROUP_CHANGED" ]
onEvent[ "EVENT_GROUP_DISAPPEARED" ] = onEvent[ "EVENT_GROUP_CHANGED" ]
onEvent[ "EVENT_GROUP_MEMBER_ADDED" ] = onEvent[ "EVENT_GROUP_CHANGED" ]
onEvent[ "EVENT_GROUP_MEMBER_REMOVED" ] = onEvent[ "EVENT_GROUP_CHANGED" ]
onEvent[ "EVENT_GROUP_MEMBER_STATUS_CHANGED" ] = onEvent[ "EVENT_GROUP_CHANGED" ]
onEvent[ "EVENT_RAID_CHANGED" ] = onEvent[ "EVENT_GROUP_CHANGED" ]
onEvent[ "EVENT_RAID_APPEARED" ] = onEvent[ "EVENT_GROUP_CHANGED" ]
onEvent[ "EVENT_RAID_DISAPPEARED" ] = onEvent[ "EVENT_GROUP_CHANGED" ]
onEvent[ "EVENT_RAID_MEMBER_STATUS_CHANGED" ] = onEvent[ "EVENT_GROUP_CHANGED" ]
--------------------------------------------------------------------------------
-- Party / Raid member position updated
onEvent[ "EVENT_PARTY_MEMBER_POSITION_UPDATED" ] = function( params )
	local Geodata = ( Mode == enumMode.Standard ) and cartographer.GetObjectGeodata( params.id, MapID ) or MapGeodata
	Minimap.UpdateMarkerPosition( Markers.Party[ params.id ], Geodata )
end
onEvent[ "EVENT_RAID_MEMBER_POSITION_UPDATED" ] = onEvent[ "EVENT_PARTY_MEMBER_POSITION_UPDATED" ]
--------------------------------------------------------------------------------
-- Spouse position updated
onEvent[ "EVENT_SPOUSE_POSITION_UPDATED" ] = function( params )
	local Geodata = Minimap.ForceGetObjectGeodata( params.id )
	Minimap.UpdateMarkerPosition( Markers.Spouse[ params.id ], Geodata )
end
--------------------------------------------------------------------------------
-- Track detected
onEvent[ "EVENT_TRACK_ADDED" ] = function( params )
	local ti = objects.GetTrackInfo( params.id )
	if ti and ti.position then
		--Minimap.CheckBounds( ti.position, ? ) -- TODO: Requires accurate testing.
		ti.text = StripHtmlTags( ti.text )
		Minimap.CreateMarker( "Track", params.id, ti.playerName, ti.text,  Style.Neutral, ti.image, MapGeodata, ti.position )
	end
end
--------------------------------------------------------------------------------
-- Track position updated
onEvent[ "EVENT_TRACK_POS_CHANGED" ] = function( params )
	local ti = objects.GetTrackInfo( params.id )
	if not ti or not ti.position then return end
	--Minimap.CheckBounds( ti.position, ? ) -- TODO: Requires accurate testing.
	Minimap.UpdateMarkerPosition( Markers.Track[ params.id ], MapGeodata, ti.position )
end
--------------------------------------------------------------------------------
-- Track lost
onEvent[ "EVENT_TRACK_REMOVED" ] = function( params )
	Minimap.DestroyMarker( "Track", params.id )
end
--------------------------------------------------------------------------------
-- Transport detected
onEvent[ "EVENT_TRANSPORT_SPAWNED" ] = function( params )
	local Pos = transport.GetPosition( params.id )
	local Dir = transport.GetDirection( params.id )
	local Name = object.GetName( params.id )
	Name = common.IsEmptyWString( Name ) and AoMapResources:L( "Transport" ) or Name
	if Pos and Dir and Name then
		--Minimap.CheckBounds( Pos, ? ) -- TODO: Requires accurate testing.
		Minimap.CreateMarker( "Transport", params.id, Name, nil,  Style.Neutral, nil, MapGeodata, Pos, Dir )
	end
end
--------------------------------------------------------------------------------
-- Transport position updated
onEvent[ "EVENT_TRANSPORT_POS_CHANGED" ] = function( params )
	local Pos = transport.GetPosition( params.id )
	local Dir = transport.GetDirection( params.id )
	if not Pos or not Dir then return end
	--Minimap.CheckBounds( Pos, ? ) -- TODO: Requires accurate testing.
	Minimap.UpdateMarkerPosition( Markers.Transport[ params.id ], MapGeodata, Pos, Dir )
end
--------------------------------------------------------------------------------
-- Transport lost
onEvent[ "EVENT_TRANSPORT_DESPAWNED" ] = function( params )
	Minimap.DestroyMarker( "Transport", params.id )
end
--------------------------------------------------------------------------------
-- SOS detected
onEvent[ "EVENT_SOS_STARTED" ] = function( params ) -- TODO: It's not tested.
	local si = cartographer.GetSosInfo( params.id )
	if not si or si.zonesMapId ~= MapID then return end
	si.message = StripHtmlTags( si.message )
	Minimap.CreateMarker( "SOS", params.id, userMods.ToWString( "SOS" ), si.message, Style.Friend, nil, MapGeodata, si.position )
end
--------------------------------------------------------------------------------
-- SOS lost
onEvent[ "EVENT_SOS_FINISHED" ] = function( params )
	Minimap.DestroyMarker( "SOS", params.id )
end
--------------------------------------------------------------------------------
-- Avatar position updated
onEvent[ "EVENT_AVATAR_POS_CHANGED" ] = function( params )
	local GP = object.GetPos( avatar.GetId() )
	if not GP then return end
	Minimap.CheckBounds( GP )
	MapPlacement.posX = math.floor( 0 - ( GP.posX - MapGeodata.x ) / MapMPP.X + MapRadius )
	MapPlacement.posY = math.floor( 0 - ZoomSize[Zoom].Y + ( GP.posY - MapGeodata.y ) / MapMPP.Y + MapRadius )
	wtMap:SetPlacementPlain( MapPlacement )
	wtMarkers:SetPlacementPlain( MapPlacement )
	Minimap.ApplyCircleMask()
end
--------------------------------------------------------------------------------
-- Avatar direction changed
onEvent[ "EVENT_AVATAR_DIR_CHANGED" ] = function( params )
	local angle = avatar.GetDir()
	wtPlayerArrow:Rotate( angle )
	local angle = math.mod( angle + math.pi / 4, math.pi * 2 )
	local alpha = - ( math.cos( angle ) - 1 ) / 2
	wtPlayerArrow:SetForegroundColor( { a = alpha, r = 1, g = 1, b = 1 } )
end
--------------------------------------------------------------------------------
-- Recalculating map Geodata (works in Unknown instances ONLY)
onIncognita[ "EVENT_SECOND_TIMER" ] = function( params )
	if not MapExplore then return end
	-- Instance Bounds backup
	for i = math.max( iSafeEnter, iSafeQuit ), 1, -1 do
		MapExplore.B[ i ] = MapExplore.B[ i - 1 ]
	end
	MapExplore.B[ 0 ] = {}
	for i,v in MapExplore.B[ 1 ] do MapExplore.B[ 0 ][ i ] = v end
	MapExplore.B[ 0 ].Dirty = not MapExplore.B[ iSafeEnter ] and true or nil
	-- Refreshing map, if requested
	if MapExplore.Altered then
		Minimap.RecalculateGeodata()
		onBase[ "EVENT_AVATAR_CLIENT_ZONE_CHANGED" ]()
	end
end
--------------------------------------------------------------------------------
-- Widgets' visual effects finish
onBase[ "EVENT_EFFECT_FINISHED" ] = function( params )
	if params.wtOwner:IsEqual( wtZoomIn ) or params.wtOwner:IsEqual( wtZoomOut ) then
		if fFadeoutButtons then
			fFadeoutButtons = false
			wtButtons:FinishFadeEffect()
			wtButtons:Show( false )
		end
	--else
		-- TODO: SOS marker blinking
	end
end
--------------------------------------------------------------------------------
-- AddonManager (dev preview) addon info request
onBase[ "SCRIPT_ADDON_INFO_REQUEST" ] = function( params )
	if params.target ~= common.GetAddonName() then return end
	local D = {
		rus = "   \" \".",
		eng = "Decent minimap for \"Allods Online\".",
		ger = "Schicklich Minikarte fr \"Allods Online\".",
		fra = "Minicarte dcent pour \"Allods Online\".",
		br  = "Minimapa decente para \"Allods Online\".",
		jpn = "AbYIC܂Ƃȃ~j}bvB",
		chn = "Decent minimap for \"Allods Online\".",
	}
	userMods.SendEvent( "SCRIPT_ADDON_INFO_RESPONSE", {
		sender = params.target,
		desc = D[ GetGameLocalization() ] or D.eng,
		addonsBlocked = { "Minimap" },
	} )
end
--------------------------------------------------------------------------------
-- AddonManager (dev preview) memory usage request
onBase[ "SCRIPT_ADDON_MEM_REQUEST" ] = function( params )
	if params.target ~= common.GetAddonName() then return end
	userMods.SendEvent( "SCRIPT_ADDON_MEM_RESPONSE",
		{ sender = params.target, memUsage = gcinfo() } )
end
--------------------------------------------------------------------------------
-- AddonManager (old version) memory usage request
onBase[ "U_EVENT_ADDON_MEM_USAGE_REQUEST" ] = function( params )
	userMods.SendEvent( "U_EVENT_ADDON_MEM_USAGE_RESPONSE",
		{ sender = common.GetAddonName(), memUsage = gcinfo() } )
end
--------------------------------------------------------------------------------
-- ALT+Z & ESC Hiding (pre-2.0.07 AO versions)
onBase[ "SCRIPT_TOGGLE_UI" ] = function( params )
	mainForm:Show( params.visible )
end
--------------------------------------------------------------------------------
-- FUNCTIONS
--------------------------------------------------------------------------------
function Minimap.CreateMarker( strLayer, ID, wsName, wsDesc, strStyle, Texture, Geodata, GamePosition, Direction )
	if Markers[ strLayer ][ ID ] then
		Minimap.DestroyMarker( strLayer, ID )
	end
	Markers[ strLayer ][ ID ] = {}
	local Marker = Markers[ strLayer ][ ID ]
	Marker.ID = ID
	if type( wsName ) ~= "table" then -- WString or nil
		Marker.Name = { wsName or userMods.ToWString( "?" ) }
		Marker.Desc = { wsDesc or common.GetEmptyWString() }
	else -- Table of WStrings/false's (used only for Quest location markers)
		Marker.Name = {}
		Marker.Desc = {}
		for i in wsName do
			table.insert( Marker.Name, wsName[ i ] or userMods.ToWString( "?" ) )
			table.insert( Marker.Desc, wsDesc[ i ] or common.GetEmptyWString() )
		end
	end
	Marker.Style = strStyle
	Marker.Widget = mainForm:CreateWidgetByDesc( wtMarkerDESC )
	wtMarkers:AddChild( Marker.Widget )
	if not Texture then
		Texture = strLayer
	end
	if type( Texture ) == "string" then -- Texture name passed
		if MarkersPriority[ Texture ] then -- Do I have this icon?
			Marker.Widget:SetPriority( MarkersPriority[ Texture ] )
			Texture = common.GetAddonRelatedTexture( Texture )
		else
			Marker.Widget:SetPriority( MarkersPriority[ strLayer ] or 1 )
			Texture = common.GetAddonRelatedTexture( "Placeholder" )
		end
	else -- Texture image (userdata) passed instead of its name (used only for Track and MapMark markers)
		Marker.Widget:SetPriority( MarkersPriority[ strLayer ] or MarkersPriority.Placeholder )
	end
	Marker.Texture = Texture
	local tsize = widgetsSystem.GetTextureSize( Texture )
	local p = Marker.Widget:GetPlacementPlain()
	local iWidgetScale = MarkersScale[ strLayer ] or MarkersScale.Default
	p.sizeX = math.floor( tsize.sizeX * iWidgetScale )
	p.sizeY = math.floor( tsize.sizeY * iWidgetScale )
	Marker.Widget:SetPlacementPlain( p )
	Marker.Widget:SetBackgroundTexture( Texture )
	Minimap.UpdateMarkerPosition( Marker, Geodata, GamePosition, Direction )
	-- Mouse catcher
	local wtMouseCatcher = mainForm:CreateWidgetByDesc( wtMouseCatcherDESC )
	Marker.Widget:AddChild( wtMouseCatcher )
	wtMouseCatcher:SetName( "Mark" .. tostring( ID ) )
	local mcp = wtMouseCatcher:GetPlacementPlain()
	local iActiveAreaScale = ActiveAreaScale[ strLayer ] or ActiveAreaScale.Default
	mcp.sizeX = math.ceil( p.sizeX * iActiveAreaScale )
	mcp.sizeY = math.ceil( p.sizeY * iActiveAreaScale )
	wtMouseCatcher:SetPlacementPlain( mcp )
	-- Party/Raid members needs fancy coloring :)
	if strLayer == "Party" then
		local C = unit.GetClass( ID )
		C = C and C.className or "UNKNOWN"
		Marker.Widget:SetBackgroundColor( ClassColors[ C ] or ClassColors[ "UNKNOWN" ] )
	end
end
--------------------------------------------------------------------------------
function Minimap.UpdateMarkerPosition( Marker, Geodata, GamePosition, Direction )
	-- If Geodata is nil, then marker will be hidden, because Geodata=nil means
	-- player is on another map (standard mode) or is NOT in instance (instance mode).
	if not Marker then return end
	if Geodata and not GamePosition then
		GamePosition = object.GetPos( Marker.ID )
	end
	if not Geodata or not GamePosition then
		Marker.Ignored = true
		Marker.Widget:Show( false )
		return
	end
	Marker.Ignored = nil
	--Marker.Widget:Show( true ) -- ApplyCircleMask() does this dirty job.
	-- Placing marker
	if Direction then
		Marker.Widget:Rotate( Direction )
	end
	local p = Marker.Widget:GetPlacementPlain()
	p.posX = math.floor( ( GamePosition.posX - Geodata.x ) / MapMPP.X - p.sizeX / 2 )
	p.posY = math.floor( ZoomSize[Zoom].Y - ( GamePosition.posY - Geodata.y ) / MapMPP.Y - p.sizeY / 2 )
	Marker.Widget:SetPlacementPlain( p )
	Minimap.ApplyCircleMask( Marker, p )
	-- Highlighter movement
	if HoveredMarker == Marker.ID then
		if Direction then
			wtHighlighter:Rotate( Direction )
		end
		wtHighlighter:SetPlacementPlain( p )
	end
end
--------------------------------------------------------------------------------
function Minimap.DestroyMarker( strLayer, ID )
	if strLayer then
		if ID then
			if Markers[ strLayer ][ ID ] then
				Markers[ strLayer ][ ID ].Widget:DestroyWidget()
				Markers[ strLayer ][ ID ] = nil
			end
		else
			for ID in Markers[ strLayer ] do
				Minimap.DestroyMarker( strLayer, ID )
			end
		end
	else
		for L in Markers do
			Minimap.DestroyMarker( L, ID )
		end
	end
end
--------------------------------------------------------------------------------
function Minimap.ApplyCircleMask( Marker, Placement )
	if Marker then
		if Marker.Ignored then return end
		local OffsetX = 0 - MapPlacement.posX + MapRadius - Placement.posX - Placement.sizeX / 2
		local OffsetY = 0 - MapPlacement.posY + MapRadius - Placement.posY - Placement.sizeY / 2
		Marker.Widget:Show( math.sqrt( OffsetX ^ 2 + OffsetY ^ 2 ) < MapRadius + 4 - Placement.sizeX / 2 )
	else
		for L in Markers do
			for _,M in Markers[ L ] do
				Minimap.ApplyCircleMask( M, M.Widget:GetPlacementPlain() )
			end
		end
	end
end
--------------------------------------------------------------------------------
function Minimap.ForceGetObjectGeodata( ID, Pos )
	if not object.IsExist( ID ) or not object.IsUnit( ID ) or not unit.IsPlayer( ID ) then return end
	if not Pos then
		Pos = object.GetPos( ID )
		if not Pos then return end
	end
	if Mode == enumMode.Standard then
		return cartographer.GetObjectGeodata( ID, MapID )
	else
		if Pos.posX < MapGeodata.x or Pos.posY < MapGeodata.y
		or Pos.posX > MapGeodata.x + MapGeodata.width
		or Pos.posY > MapGeodata.y + MapGeodata.height then
			return nil
		end
		return MapGeodata
	end
end
--------------------------------------------------------------------------------
function Minimap.CheckBounds( Pos, iMaxDistance )
	if Mode ~= enumMode.Exploring or not Pos then return end
	if iMaxDistance then
		local PP = object.GetPos( avatar.GetId() )
		if not PP then return end
		local DistX = math.max( PP.posX, Pos.posX ) - math.min( PP.posX, Pos.posX )
		local DistY = math.max( PP.posY, Pos.posY ) - math.min( PP.posY, Pos.posY )
		if DistX > iMaxDistance or DistY > iMaxDistance then return end
	end
	if Pos.posX < MapExplore.B[0].MinX or Pos.posX > MapExplore.B[0].MaxX
	or Pos.posY < MapExplore.B[0].MinY or Pos.posY > MapExplore.B[0].MaxY then
		MapExplore.B[0].MinX = math.min( MapExplore.B[0].MinX, Pos.posX )
		MapExplore.B[0].MaxX = math.max( MapExplore.B[0].MaxX, Pos.posX )
		MapExplore.B[0].MinY = math.min( MapExplore.B[0].MinY, Pos.posY )
		MapExplore.B[0].MaxY = math.max( MapExplore.B[0].MaxY, Pos.posY )
		MapExplore.Altered = true
	end
end
--------------------------------------------------------------------------------
function Minimap.RecalculateGeodata()
	if Mode ~= enumMode.Exploring then return end
	MapExplore.Altered = false
	local InstanceW = MapExplore.B[0].MaxX - MapExplore.B[0].MinX
	local InstanceH = MapExplore.B[0].MaxY - MapExplore.B[0].MinY
	local SafeW = ZoomSize[ 1 ].X - MapRadius * 2
	local SafeH = ZoomSize[ 1 ].Y - MapRadius * 2
	local MPP = math.max( InstanceW, InstanceH * SafeW / SafeH ) / SafeW
	MapGeodata.width  = MPP * ZoomSize[ 1 ].X
	MapGeodata.height = MPP * ZoomSize[ 1 ].Y
	MapGeodata.x = MapExplore.B[0].MinX - ( MapGeodata.width  - InstanceW ) / 2
	MapGeodata.y = MapExplore.B[0].MinY - ( MapGeodata.height - InstanceH ) / 2
end
--------------------------------------------------------------------------------
function StripHtmlTags( wsText )
	if not wsText or not common.IsWString( wsText ) then return common.GetEmptyWString() end
	local strText = userMods.FromWString( wsText )
	strText = string.gsub( strText, "</?%w[%w%s=\"]-/?>", "" ) -- Strips html tags.
	strText = string.gsub( strText, "^%s*(.-)%s*$", "%1" ) -- Trims spaces.
	return userMods.ToWString( strText )
end
--------------------------------------------------------------------------------
function EventHandlersRegistration( Enable, Handlers )
	for event, handler in Handlers do
		if Enable then
			common.RegisterEventHandler( handler, event )
		else
			common.UnRegisterEventHandler( handler, event )
		end
	end
end
--------------------------------------------------------------------------------
-- INITIALIZATION
--------------------------------------------------------------------------------
function Init()
	-- Widgets (indents shows hierarchy)
	wtTooltip = mainForm:GetChildChecked( "wtTooltip", false )
	wtMainPanel = mainForm:GetChildChecked( "wtMainPanel", false )
		wtButtons = wtMainPanel:GetChildChecked( "wtButtons", false )
			wtZoomIn = wtButtons:GetChildChecked( "wtZoomIn", false )
				btnZoomIn = wtZoomIn:GetChildChecked( "btnZoomIn", false )
			wtZoomOut = wtButtons:GetChildChecked( "wtZoomOut", false )
				btnZoomOut = wtZoomOut:GetChildChecked( "btnZoomOut", false )
		wtPointers = wtMainPanel:GetChildChecked( "wtPointers", false )
			wtPlayerArrow = wtPointers:GetChildChecked( "wtPlayerArrow", false )
			wtMarkers = wtPointers:GetChildChecked( "wtMarkers", false )
				wtHighlighter = wtMarkers:GetChildChecked( "wtHighlighter", false )
				local wtMarker = wtMarkers:GetChildChecked( "wtMarker", false )
					local wtMouseCatcher = wtMarker:GetChildChecked( "wtMouseCatcher", false )
					wtMouseCatcherDESC = wtMouseCatcher:GetWidgetDesc()
					wtMouseCatcher:DestroyWidget()
				wtMarkerDESC = wtMarker:GetWidgetDesc()
				wtMarker:DestroyWidget()
		wtCircle = wtMainPanel:GetChildChecked( "wtCircle", false )
			wtMap = wtCircle:GetChildChecked( "wtMap", false )
				wtInstance = wtMap:GetChildChecked( "wtInstance", false )
		wtTitle = wtMainPanel:GetChildChecked( "wtTitle", false )
			labTitle = wtTitle:GetChildChecked( "labTitle", false )
	
	-- Reactions & Events
	for name, handler in onReaction do
		common.RegisterReactionHandler( handler, name )
	end
	EventHandlersRegistration( true, onBase )
	
	-- Turning off built-in controls under Minimap
	common.StateUnloadManagedAddon( "Minimap" )
	--common.StateUnloadManagedAddon( "ContextPOIMarker" )
	-- TODO: Use it! See avatar.IsClientDetectorEnabled() and avatar.EnableClientDetector()
	
	-- Getting some constants
	MapRadius = math.floor( wtCircle:GetPlacementPlain().sizeX / 2 )
	MapPlacement = wtMap:GetPlacementPlain()
	
	-- Calculating Zoom sizes
	ZoomSize[ 1 ] = { X = MapPlacement.sizeX, Y = MapPlacement.sizeY }
	for i = 2, 4 do
		ZoomSize[ i ] = {}
		ZoomSize[ i ].X = math.ceil( ZoomSize[ i - 1 ].X * 1.6 )
		ZoomSize[ i ].Y = math.ceil( ZoomSize[ i - 1 ].Y * 1.6 )
	end
	
	-- Checking if AoMapResources loaded successfully.
	if rawget( _G, "AoMapResources" ) == nil then
		Global( "AoMapResources", {
			InstanceMapFound = function() end,
			GetInstanceMap = function() end,
			GetKnownObjects = function() end,
			L = function( self, s ) return userMods.ToWString( s ) end,
		} )
	end
	if rawget( _G, "GetGameLocalization" ) == nil then
		Global( "GetGameLocalization", function() return "eng" end )
	end
	if AoMapResources.CheckFormat then
		AoMapResources:CheckFormat( 1 )
	end
	
	-- Cleaning and fixing saved data
	local cfg = userMods.GetGlobalConfigSection( common.GetAddonName() ) or {}
	local fAltered
	if not cfg.CfgVersion or cfg.CfgVersion < 2 then
		cfg = { CfgVersion = 2 }
		fAltered= true
	end
	if cfg.Bounds then
		local iCount = 0
		for Z in cfg.Bounds do
			if AoMapResources:InstanceMapFound( false, Z, "Default" ) then
				cfg.Bounds[ Z ] = nil
				fAltered= true
			else
				iCount = iCount + 1
			end
		end
		if iCount == 0 then
			cfg.Bounds = nil
		end
	end
	if fAltered then
		userMods.SetGlobalConfigSection( common.GetAddonName(), cfg )
	end
	
	-- Special fix for Japanese AO version (game bug?)
	if rawget( _G, "INTERACTION_CLASS_RELATED" ) == nil then
		Global( "INTERACTION_CLASS_RELATED", 3 )
	end
	
	-- Backward compatibility with pre-1.1.04 versions (China, Japan, etc.):
	if not social.GetFriendInfo then
		mainForm:SetPriority( 980 )
	end
	
	-- For the case if addon was reloaded by AddonManager
	if avatar.IsExist() then
		onBase[ "EVENT_AVATAR_CREATED" ]()
	end
end
--------------------------------------------------------------------------------
Init()
--------------------------------------------------------------------------------
